module net.BurtonRadons.dig.platform.menu;

private import net.BurtonRadons.dig.platform.base;
private import net.BurtonRadons.dig.platform.control;
private import net.BurtonRadons.dig.platform.frame;
private import net.BurtonRadons.dig.platform.windows;

//private import std.c.windows.windows;


/** A menu pseudo-widget.  Like the FileSelector, this is a class that you
  * construct off-line before running on-line (interrupting the thread) and
  * then retrieving the value.
  *
  * The following partial example creates a standard New/Open/Close menu
  * for a text editor:
  *
  * @code

class MenuFile : Menu
{
    Program program; // Parent running program

    this (Program program)
    {
        bit document = (program.currentDocument === null);

        super (program);
        this.program = program;
        add ("New...", &New);
        add ("Open...", &Open);
        add ("Close", &Close); disabled (!document);
        separator ();
        add ("Save", &Save); disabled (!document);
        add ("Save As...", &SaveAs); disabled (!document);
        add ("Save All", &SaveAll); disabled (!program.documents.length);
        separator ();
        add ("Exit", &Exit);
        exec ();
    }

    void New     (Event e) { program.createDocument (); }
    void Open    (Event e) { program.openDocument (); }
    void Close   (Event e) { program.closeDocument (); }
    void Save    (Event e) { program.saveDocument (); }
    void SaveAs  (Event e) { program.saveDocumentAs (); }
    void SaveAll (Event e) { program.saveAllDocuments (); }
    void Exit    (Event e) { program.exit (); }
}

  * @endcode
  */

class Menu
{
    /** Create an empty popup menu. */
    this (Control control)
    {
        digPlatformControl = control;
        digPlatformHandle = CreatePopupMenu ();
    }

    /** Destroy the menu. */
    ~this ()
    {
        if (digPlatformHandle)
            DestroyMenu (digPlatformHandle);
        digPlatformHandle = (_HANDLE) 0;
    }

    /** Add a textual menu item, get a dispatcher. */
    Dispatcher *add (char [] name, char [] text)
    {
        Dispatcher dispatcher;

        digPlatformDispatchers ~= dispatcher;

        digPlatformLastItem ++;
        if (!AppendMenuA (digPlatformHandle, MF_STRING, digPlatformDispatchers.length, std.string.toStringz (text)))
            digPlatformGetLastError ("Menu.add");

        if (name !== null)
            digPlatformNameToID [name] = digPlatformLastItem;
        return &digPlatformDispatchers [digPlatformDispatchers.length - 1];
    }

    /** Add a textual menu item, get a dispatcher. */
    Dispatcher *add (char [] text)
    {
        return add (null, text);
    }

    /** Add a textual menu item, add a dispatcher command, return the dispatcher. */
    Dispatcher *add (char [] name, char [] text, Dispatcher.Method method)
    {
        Dispatcher *dispatcher = add (name, text);

        dispatcher.add (method);
        return dispatcher;
    }

    /** Add a textual menu item, add a dispatcher command, return the dispatcher. */
    Dispatcher *add (char [] name, char [] text, Dispatcher.MethodB method)
    {
        Dispatcher *dispatcher = add (name, text);

        dispatcher.add (method);
        return dispatcher;
    }

    /** Add a textual menu item, add a dispatcher command, return the dispatcher. */
    Dispatcher *add (char [] text, Dispatcher.Method method)
    {
        return add (null, text, method);
    }

    /** Add a textual menu item, add a dispatcher command, return the dispatcher. */
    Dispatcher *add (char [] text, Dispatcher.MethodB method)
    {
        return add (null, text, method);
    }

    /** Set whether the most recently added item should be disabled. */
    void disabled (bit value) { digPlatformDisabled (digPlatformLastItem, value); }

    /** Set whether the named item should be disabled. */
    void disabled (char [] name, bit value) { digPlatformDisabled (digPlatformNameToID [name], value); }

    /** Set whether to highlight the most recently added item. */
    void highlight (bit value) { digPlatformHighlight (digPlatformLastItem, value); }

    /** Set whether to highlight the named item. */
    void highlight (char [] name, bit value) { digPlatformHighlight (digPlatformNameToID [name], value); }

    /** Set whether a checkmark appears beside the most recently added item. */
    void checked (bit value) { digPlatformChecked (digPlatformLastItem, value); }

    /** Set whether a checkmark appears beside the named item. */
    void checked (char [] name, bit value) { digPlatformChecked (digPlatformNameToID [name], value); }

    /** Add a popup text item. */
    void popup (char [] name, char [] text, Menu menu)
    {
        digPlatformLastItem ++;
        digPlatformForever [menu] = true;
        AppendMenuA (digPlatformHandle, MF_STRING | MF_POPUP, (_DWORD) menu.digPlatformHandle, std.string.toStringz (text));
        if (name !== null)
            digPlatformNameToID [name] = digPlatformLastItem;
    }
    
    /** Add a popup menu text item. */
    void popup (char [] text, Menu menu)
    {
        popup (null, text, menu);
    }

    /** Add a separator. */
    void separator ()
    {
        digPlatformLastItem ++;
        AppendMenuA (digPlatformHandle, MF_SEPARATOR, 0, null);
    }

    /** Display the menu and don't return until a command has been executed or it has been dismissed. */
    void exec ()
    {
        _POINT point;

        if (!GetCursorPos (&point))
            digPlatformGetLastError ("Menu.exec <GetCursorPos>");

        int x = point.x;
        int y = point.y;
        
        int result = TrackPopupMenu (digPlatformHandle, TPM_RETURNCMD, x, y, 0, digPlatformControl !== null ? digPlatformControl.digPlatformHWND : (_HANDLE) 0, null);

        if (!result)
            return;
        if (result <= digPlatformDispatchers.length && result >= 0)
            digPlatformDispatchers [result - 1].notify ();
    }
    
/+
#ifdef DoxygenShouldSkipThis
+/

    void digPlatformActivate (int index)
    {
        digPlatformDispatchers [index - 1].notify ();
    }
    
    _HMENU digPlatformHandle;
    bit [Menu] digPlatformForever;
    static Dispatcher [] digPlatformDispatchers; /* Dispatcher for each line */
    int [char []] digPlatformNameToID; /* Translates from an item name to an identifier */

    int digPlatformLastItem = -1;
    Control digPlatformControl;
    
    void digPlatformDisabled (int id, bit value)
    {
        MENUITEMINFO info;

        info.cbSize = info.size;
        info.fMask |= MIIM_STATE;
        info.fState |= value ? MFS_DISABLED : MFS_ENABLED;
        if (!SetMenuItemInfoA (digPlatformHandle, id, true, &info))
            digPlatformGetLastError ("Menu.disabled (post)");
    }

    void digPlatformHighlight (int id, bit value)
    {
        MENUITEMINFO info;

        info.cbSize = info.size;
        info.fMask |= MIIM_STATE;
        info.fState |= value ? MFS_HILITE : MFS_UNHILITE;
        if (!SetMenuItemInfoA (digPlatformHandle, id, true, &info))
            digPlatformGetLastError ("Menu.disabled (post)");
    }

    void digPlatformChecked (int id, bit value)
    {
        MENUITEMINFO info;

        info.cbSize = info.size;
        info.fMask |= MIIM_STATE;
        info.fState |= value ? MFS_CHECKED : MFS_UNCHECKED;
        if (!SetMenuItemInfoA (digPlatformHandle, id, true, &info))
            digPlatformGetLastError ("Menu.disabled (post)");
    }

    this (char [] dontdoit) { }
    
/+
#endif
+/
}

/** A menu bar for frames.  This should be considered interim;
  * it will be removed in the future.
  */

class MenuBar : Menu
{
    /** Create the menu bar. */
    this ()
    {
        super ("dontcreatethemenu");
        digPlatformHandle = CreateMenu ();
    }
}
